<?php
require_once 'config.php';
require_admin();

$message = '';
$selectedPackId = $_GET['pack_id'] ?? null;

// --- CRUD Operations ---
if (isset($_POST['add_lesson'])) {
    $packId = (int)$_POST['pack_id'];
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $youtubeLink = trim($_POST['youtube_link']);
    $thumbnailPath = null; // Initialize thumbnail path

    // 1. Validate YouTube Link
    if (!get_youtube_embed_link($youtubeLink)) {
        $message = 'Error: The provided YouTube link is invalid.';
    } else {
        // 2. Custom Thumbnail Upload Handling
        if (isset($_FILES['custom_thumbnail']) && $_FILES['custom_thumbnail']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = 'uploads/tumbs/';

            // Ensure the upload directory exists
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }

            $fileTmpPath = $_FILES['custom_thumbnail']['tmp_name'];
            $fileName = $_FILES['custom_thumbnail']['name'];
            $fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

            $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif'];
            $maxFileSize = 5 * 1024 * 1024; // 5MB

            if (!in_array($fileExtension, $allowedExtensions)) {
                $message = 'Error: Invalid file type for thumbnail. Only JPG, JPEG, PNG, and GIF are allowed.';
            } elseif ($_FILES['custom_thumbnail']['size'] > $maxFileSize) {
                $message = 'Error: The uploaded file is too large. Maximum size is 5MB.';
            } else {
                // Generate a unique filename
                $newFileName = uniqid('tumb_') . '.' . $fileExtension;
                $destPath = $uploadDir . $newFileName;

                if (move_uploaded_file($fileTmpPath, $destPath)) {
                    $thumbnailPath = $destPath; // Save the relative path
                } else {
                    $message = 'Error moving the uploaded thumbnail file. Check folder permissions.';
                }
            }
        }
        
        // 3. Database Insertion (only proceed if no file upload error occurred or if no file was uploaded)
        if (strpos($message, 'Error') === false) {
            try {
                // INSERT statement updated to include thumbnail_path
                $stmt = $pdo->prepare("INSERT INTO lessons (pack_id, title, description, youtube_link, thumbnail_path) VALUES (?, ?, ?, ?, ?)");
                $stmt->execute([$packId, $title, $description, $youtubeLink, $thumbnailPath]);
                $message = 'Lesson added successfully to the pack!';
                $selectedPackId = $packId; // Keep the selected pack
            } catch (PDOException $e) {
                $message = 'Error adding lesson: ' . $e->getMessage();
            }
        }
    }
}

// --- Fetch all Lesson Packs ---
$stmt = $pdo->query("SELECT id, title FROM lesson_packs ORDER BY title");
$packs = $stmt->fetchAll();

// --- Fetch Lessons for Selected Pack (for viewing/managing) ---
$lessons = [];
$packTitle = 'No Pack Selected';
if ($selectedPackId) {
    $stmt = $pdo->prepare("SELECT title FROM lesson_packs WHERE id = ?");
    $stmt->execute([$selectedPackId]);
    $packTitle = $stmt->fetchColumn() ?: 'Unknown Pack';

    // SELECT statement updated to include thumbnail_path
    $stmt = $pdo->prepare("SELECT id, title, description, youtube_link, thumbnail_path FROM lessons WHERE pack_id = ? ORDER BY id DESC");
    $stmt->execute([$selectedPackId]);
    $lessons = $stmt->fetchAll();
}

require_once 'layouts.php';
render_header('Add Lesson');
render_sidebar($_SESSION['user_role'], basename(__FILE__));
render_topbar('Add Lesson to Pack');
?>

<div class="space-y-8">
    <?php if ($message): ?>
        <div class="bg-indigo-100 border-l-4 border-indigo-500 text-indigo-700 p-4 mb-6 rounded-lg" role="alert"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <!-- Add New Lesson Form -->
    <div class="bg-white p-6 rounded-xl shadow-lg">
        <h2 class="text-2xl font-bold text-gray-900 mb-6">Create New Lesson</h2>
        <!-- IMPORTANT: Added enctype="multipart/form-data" -->
        <form method="POST" action="addclass.php" class="space-y-4" enctype="multipart/form-data">
            <div>
                <label for="pack_id" class="block text-sm font-medium text-gray-700">Select Lesson Pack</label>
                <select name="pack_id" id="pack_id" required class="mt-1 block w-full px-4 py-2 border border-gray-300 rounded-lg" onchange="window.location.href='addclass.php?pack_id=' + this.value">
                    <option value="">-- Select a Pack --</option>
                    <?php foreach ($packs as $pack): ?>
                        <option value="<?php echo $pack['id']; ?>" <?php echo ($pack['id'] == $selectedPackId) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($pack['title']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <?php if ($selectedPackId): ?>
            <div>
                <label for="title" class="block text-sm font-medium text-gray-700">Lesson Title</label>
                <input type="text" name="title" id="title" required class="mt-1 block w-full px-4 py-2 border border-gray-300 rounded-lg">
            </div>
            <div>
                <label for="description" class="block text-sm font-medium text-gray-700">Lesson Description</label>
                <textarea name="description" id="description" rows="3" required class="mt-1 block w-full px-4 py-2 border border-gray-300 rounded-lg"></textarea>
            </div>
            
            <!-- NEW: Custom Thumbnail Upload Field -->
            <div>
                <label for="custom_thumbnail" class="block text-sm font-medium text-gray-700">Custom Thumbnail Image (Optional)</label>
                <input type="file" name="custom_thumbnail" id="custom_thumbnail" accept="image/png, image/jpeg, image/gif" class="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-indigo-50 file:text-indigo-700 hover:file:bg-indigo-100">
                <p class="mt-1 text-xs text-gray-500">Max 5MB. Accepts JPG, JPEG, PNG, GIF.</p>
            </div>
            
            <!-- Original YouTube Link Field -->
            <div>
                <label for="youtube_link" class="block text-sm font-medium text-gray-700">YouTube Embed Video Link (Full URL)</label>
                <input type="url" name="youtube_link" id="youtube_link" placeholder="e.g., https://www.youtube.com/watch?v=dQw4w9WgXcQ" required
                        class="mt-1 block w-full px-4 py-2 border border-gray-300 rounded-lg" oninput="previewThumbnail(this.value)">
                <div id="thumbnail-preview" class="mt-2 h-48 w-full bg-gray-200 rounded-lg flex items-center justify-center text-gray-500 overflow-hidden">
                    Video Thumbnail Preview
                </div>
            </div>
            <button type="submit" name="add_lesson" class="w-48 py-3 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 font-semibold transition duration-150 shadow-md">
                Submit Lesson
            </button>
            <?php endif; ?>
        </form>
    </div>

    <!-- Lessons in Selected Pack -->
    <?php if ($selectedPackId && $lessons): ?>
    <h2 class="text-2xl font-bold text-gray-900 pt-4 border-t mt-8">Lessons in: <?php echo htmlspecialchars($packTitle); ?></h2>
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <?php foreach ($lessons as $lesson): ?>
            <?php 
                // Logic to prioritize custom thumbnail over YouTube thumbnail
                $thumbnailSrc = !empty($lesson['thumbnail_path'])
                    ? htmlspecialchars($lesson['thumbnail_path'])
                    : get_youtube_thumbnail($lesson['youtube_link']);
            ?>
            <div class="bg-white p-4 rounded-xl shadow-lg flex space-x-4">
                <img src="<?php echo $thumbnailSrc; ?>" alt="Lesson Thumbnail" class="w-28 h-20 object-cover rounded-lg flex-shrink-0">
                <div class="flex-grow">
                    <h4 class="text-lg font-bold text-gray-900 line-clamp-1"><?php echo htmlspecialchars($lesson['title']); ?></h4>
                    <p class="text-sm text-gray-600 line-clamp-2"><?php echo htmlspecialchars($lesson['description']); ?></p>
                    <span class="text-xs text-indigo-500">Lesson ID: <?php echo $lesson['id']; ?></span>
                </div>
                <!-- Delete button (omitted for brevity, but would be here) -->
            </div>
        <?php endforeach; ?>
    </div>
    <?php elseif ($selectedPackId && empty($lessons)): ?>
        <div class="bg-yellow-100 p-4 rounded-xl text-yellow-800">No lessons added to this pack yet.</div>
    <?php endif; ?>
</div>

<script>
    function previewThumbnail(url) {
        const previewDiv = document.getElementById('thumbnail-preview');
        // Simple regex to get video ID (basic implementation)
        const match = url.match(/(?:youtube\.com\/(?:[^\/]+\/.+\/|\w\/\w+\?v=)|youtu\.be\/)([^&?]+)/);
        if (match && match[1]) {
            const videoId = match[1];
            const thumbnailUrl = `https://img.youtube.com/vi/${videoId}/hqdefault.jpg`; // High quality thumbnail
            previewDiv.innerHTML = `<img src="${thumbnailUrl}" alt="Video Thumbnail" class="w-full h-full object-cover">`;
        } else {
            // Note: This script only previews YouTube links, not custom uploaded images.
            previewDiv.innerHTML = 'Video Thumbnail Preview';
        }
    }
    // Initial check if there's an existing link (e.g., on edit)
    document.addEventListener('DOMContentLoaded', () => {
        const linkInput = document.getElementById('youtube_link');
        if (linkInput && linkInput.value) {
            previewThumbnail(linkInput.value);
        }
    });
</script>

<?php render_main_content_end(); ?>