<?php
// This file assumes config.php and helpers.php are included before it.

/**
 * Renders the HTML document header, including Tailwind CSS and custom styles.
 * @param string $title The page title.
 */
function render_header($title) {
    // Load Tailwind CSS CDN and required font
    // Note: The floating-sidebar styles defined here will only apply when the element is visible (lg:flex).
    $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . htmlspecialchars($title) . ' | ' . APP_NAME . '</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: "Inter", sans-serif; background-color: #f4f7f9; }

        /* Custom styles for Floating Sidebar (Desktop View) */
        .super-shadow {
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }

        .floating-sidebar {
            position: fixed;
            top: 20px;
            left: 20px;
            height: calc(100vh - 40px);
            width: 70px;
            border-radius: 1.5rem;
            z-index: 30;
            padding: 1.5rem 0;
            transition: all 0.3s ease;
        }

        .active-gradient {
            background-image: linear-gradient(to bottom right, #6366f1, #9333ea);
        }

        .active-icon-shadow {
            box-shadow: 0 5px 15px rgba(99, 102, 241, 0.5);
        }

        /* Color classes for desktop icons */
        .icon-color-1 { background-image: linear-gradient(to right, #6366f1, #3b82f6); }
        .icon-color-2 { background-image: linear-gradient(to right, #10b981, #059669); }
        .icon-color-3 { background-image: linear-gradient(to right, #f59e0b, #d97706); }
        .icon-color-4 { background-image: linear-gradient(to right, #f43f5e, #e11d48); }
        .icon-color-5 { background-image: linear-gradient(to right, #8b5cf6, #7c3aed); }
        .icon-color-default { background-image: linear-gradient(to right, #fcae3c, #fccb6c); }
        
        /* Mobile More Menu Styling */
        #mobile-more-menu {
            transform: translateY(10px);
            opacity: 0;
            pointer-events: none;
            transition: all 0.3s ease-out;
        }
        #mobile-more-menu.open {
            transform: translateY(0);
            opacity: 1;
            pointer-events: auto;
        }
    </style>
</head>
<body>';
    echo $html;
}

/**
 * Renders the HTML document footer and includes necessary scripts.
 */
function render_footer() {
    // Simple JS to handle the mobile 'More' button pop-up
    echo '<script>
        function toggleMobileMenu() {
            const menu = document.getElementById("mobile-more-menu");
            const backdrop = document.getElementById("mobile-backdrop");
            if (menu && backdrop) {
                menu.classList.toggle("open");
                backdrop.classList.toggle("hidden");
            }
        }
        
        // Close menu when tapping outside (on the backdrop)
        document.addEventListener("DOMContentLoaded", () => {
            const backdrop = document.getElementById("mobile-backdrop");
            if (backdrop) {
                backdrop.addEventListener("click", toggleMobileMenu);
            }
        });
    </script>';
    echo '</body></html>';
}

/**
 * Renders the floating sidebar navigation (desktop) and fixed bottom navigation (mobile).
 * @param string $userRole The user role ('admin' or 'student').
 * @param string $currentPage The basename of the current file (e.g., 'profile.php').
 */
function render_sidebar($userRole, $currentPage) {
    $menu = [];
    $defaultIcon = 'fa-solid fa-square';

    if ($userRole === 'admin') {
        $menu = [
            'Dashboard' => ['url' => 'dashboard.php', 'icon' => 'fa-solid fa-chart-line', 'color' => 'icon-color-1'],
            'Lesson Packs' => ['url' => 'addpack.php', 'icon' => 'fa-solid fa-box-open', 'color' => 'icon-color-2'],
            'Add Lessons' => ['url' => 'addclass.php', 'icon' => 'fa-solid fa-graduation-cap', 'color' => 'icon-color-3'],
            'Add Notice' => ['url' => 'addnotice.php', 'icon' => 'fa-solid fa-bullhorn', 'color' => 'icon-color-4'],
            'User Management' => ['url' => 'usermanage.php', 'icon' => 'fa-solid fa-users', 'color' => 'icon-color-5'],
            'Slip Requests' => ['url' => 'sliprequests.php', 'icon' => 'fa-solid fa-receipt', 'color' => 'icon-color-default'],
        ];
    } else {
        // Student menu items
        $menu = [
            'Dashboard' => ['url' => 'studentdash.php', 'icon' => 'fa-solid fa-home', 'color' => 'icon-color-1'],
            'My Classes' => ['url' => 'myclass.php', 'icon' => 'fa-solid fa-book-open', 'color' => 'icon-color-2'],
            'Buy Classes' => ['url' => 'buyclass.php', 'icon' => 'fa-solid fa-shopping-cart', 'color' => 'icon-color-3'],
            'Notices' => ['url' => 'notices.php', 'icon' => 'fa-solid fa-bell', 'color' => 'icon-color-4'],
            'Profile' => ['url' => 'profile.php', 'icon' => 'fa-solid fa-user', 'color' => 'icon-color-5'],
        ];
    }

    $sidebarHtml = '<div class="flex bg-gray-50 min-h-screen">';

    // --- 1. Desktop Floating Sidebar (Visible on LG screens and up) ---
    // (This remains unchanged from the previous version)
    $sidebarHtml .= '
        <!-- Floating Sidebar (Desktop Only) -->
        <aside class="floating-sidebar bg-white super-shadow flex-col justify-between z-30 hidden lg:flex">
            <!-- Navigation Icons -->
            <nav class="space-y-4">';

    foreach ($menu as $name => $item) {
        $isActive = (basename($item['url']) === basename($currentPage));
        $icon = $item['icon'] ?? $defaultIcon;
        $colorClass = $item['color'] ?? 'icon-color-default';

        if ($isActive) {
            $iconBgStyle = 'active-gradient active-icon-shadow';
            $iconTextClass = 'text-white';
            $linkClasses = 'transform scale-110';
        } else {
            $iconBgStyle = $colorClass . ' shadow-md';
            $iconTextClass = 'text-white';
            $linkClasses = '';
        }

        $sidebarHtml .= '
            <a href="' . htmlspecialchars($item['url']) . '" title="' . htmlspecialchars($name) . '"
                class="flex items-center justify-center w-full transition duration-300 ' . $linkClasses . '">
                <div class="w-12 h-12 rounded-xl flex items-center justify-center ' . $iconBgStyle . '">
                    <i class="' . $icon . ' text-xl ' . $iconTextClass . '"></i>
                </div>
            </a>';
    }

    $sidebarHtml .= '
            </nav>
            <div class="flex justify-center">
                <a href="index.php?logout=true" title="Logout"
                    class="flex items-center justify-center w-12 h-12 rounded-xl bg-red-500 text-white hover:bg-red-600 transition duration-300 shadow-lg super-shadow">
                    <i class="fa-solid fa-power-off text-xl"></i>
                </a>
            </div>
        </aside>';
        
    // --- 2. Mobile Fixed Bottom Navigation (Hidden above LG screens) ---
    $mobileMenuHtml = '
        <!-- Mobile Backdrop (To close menu on tap outside) -->
        <div id="mobile-backdrop" class="fixed inset-0 bg-black bg-opacity-30 z-30 hidden lg:hidden"></div>
        
        <!-- More Menu Pop-up -->
        <div id="mobile-more-menu" class="fixed bottom-[95px] right-4 bg-white rounded-xl shadow-2xl p-3 z-40 lg:hidden super-shadow flex flex-col space-y-2 border border-gray-100">
            ';
            
    // The total number of menu items
    $totalItems = count($menu);
    // Number of items visible in the main bar (first 4 items)
    $visibleItemsCount = 4;
    
    // Split menu into visible and hidden parts
    $visibleMenu = array_slice($menu, 0, $visibleItemsCount, true);
    $hiddenMenu = array_slice($menu, $visibleItemsCount, $totalItems - $visibleItemsCount, true);
    
    // Add remaining items (hidden menu) and Logout button to the popup
    foreach ($hiddenMenu as $name => $item) {
        $isActive = (basename($item['url']) === basename($currentPage));
        $icon = $item['icon'] ?? $defaultIcon;
        $colorClass = $item['color'] ?? 'icon-color-default';

        $activeClasses = $isActive 
            ? 'active-gradient text-white font-bold'
            : 'text-gray-700 hover:bg-gray-100';

        $mobileMenuHtml .= '
            <a href="' . htmlspecialchars($item['url']) . '" title="' . htmlspecialchars($name) . '"
                class="flex items-center space-x-3 w-40 p-2 rounded-lg transition duration-200 ' . $activeClasses . '">
                <div class="w-8 h-8 rounded-full flex items-center justify-center ' . $colorClass . '">
                    <i class="' . $icon . ' text-sm text-white"></i>
                </div>
                <span class="text-sm">' . htmlspecialchars($name) . '</span>
            </a>';
    }
    
    // Add Logout to the popup menu
    $mobileMenuHtml .= '
        <a href="index.php?logout=true" title="Logout"
            class="flex items-center space-x-3 w-40 p-2 rounded-lg transition duration-200 text-red-600 hover:bg-red-50 hover:text-red-700 font-semibold">
            <div class="w-8 h-8 rounded-full flex items-center justify-center bg-red-500">
                <i class="fa-solid fa-power-off text-sm text-white"></i>
            </div>
            <span class="text-sm">Logout</span>
        </a>
    </div>';
    
    $sidebarHtml .= $mobileMenuHtml;

    // --- Main Mobile Navigation Bar ---
    $sidebarHtml .= '
        <nav class="fixed bottom-0 left-0 right-0 bg-white shadow-2xl z-40 p-3 lg:hidden border-t-4 border-indigo-600 rounded-t-2xl">
            <div class="flex justify-around items-center h-full max-w-xl mx-auto">';
    
    // 1. Render the first 4 visible menu items
    foreach ($visibleMenu as $name => $item) {
        $isActive = (basename($item['url']) === basename($currentPage));
        $icon = $item['icon'] ?? $defaultIcon;
        
        // Active item styling
        $linkClasses = $isActive 
            ? 'text-white active-gradient rounded-xl w-14 h-14 flex items-center justify-center shadow-lg active-icon-shadow transform scale-105'
            : 'text-gray-500 hover:text-indigo-600 w-14 h-14 flex items-center justify-center transition duration-200';

        $iconSize = $isActive ? 'text-2xl' : 'text-xl'; // Active icon slightly bigger

        $sidebarHtml .= '
            <a href="' . htmlspecialchars($item['url']) . '" title="' . htmlspecialchars($name) . '"
                class="flex items-center justify-center transition duration-200 ' . $linkClasses . '">
                <i class="' . $icon . ' ' . $iconSize . '"></i>
            </a>';
    }
    
    // 2. Render the "More" button
    $moreIsActive = false; // The More button itself is rarely active unless a hidden page is shown.
    foreach ($hiddenMenu as $item) {
        if (basename($item['url']) === basename($currentPage)) {
             $moreIsActive = true;
             break;
        }
    }
    
    $moreClasses = $moreIsActive 
        ? 'text-white active-gradient rounded-xl w-14 h-14 flex items-center justify-center shadow-lg active-icon-shadow transform scale-105'
        : 'text-gray-500 hover:text-indigo-600 w-14 h-14 flex items-center justify-center transition duration-200';

    $moreIconSize = $moreIsActive ? 'text-2xl' : 'text-xl';

    $sidebarHtml .= '
            <!-- More Button (Opens Pop-up) -->
            <button onclick="toggleMobileMenu()" title="More"
                class="flex items-center justify-center ' . $moreClasses . '">
                <i class="fa-solid fa-bars ' . $moreIconSize . '"></i>
            </button>';

    $sidebarHtml .= '
            </div>
        </nav>';

    // --- Main Content Start ---
    // Added lg:ml-24 for desktop space and pb-24 (padding-bottom) for mobile to prevent content from being hidden by the fixed bottom nav.
    $sidebarHtml .= '
        <main class="flex-1 overflow-x-hidden overflow-y-auto p-6 lg:p-10 z-10 relative lg:ml-24 pb-24">';
    echo $sidebarHtml;
}

/**
 * Renders the top navigation bar, including the title and user profile display.
 * @param string $title The main title for the header.
 * @param string|null $imageUrl The user's profile image URL. Defaults to null.
 */
function render_topbar($title, $imageUrl = null) {
    $currentUserName = $_SESSION['user_name'] ?? 'Guest';
    $initial = strtoupper(substr($currentUserName, 0, 1));

    // Check if a valid image URL is provided (not null and not a placeholder fallback)
    if ($imageUrl && strpos($imageUrl, 'placehold.co') === false && !empty($imageUrl)) {
        // Display the actual image
        $profileDisplayHtml = '
            <a href="profile.php" title="Welcome, ' . htmlspecialchars($currentUserName) . ' - Go to Profile">
                <img src="' . htmlspecialchars($imageUrl) . '" alt="Profile Picture" 
                    class="w-10 h-10 rounded-full object-cover border-2 border-indigo-600 cursor-pointer transition duration-300 hover:scale-105 super-shadow"
                    onerror="this.onerror=null;this.src=\'https://placehold.co/40x40/2563EB/ffffff?text=' . $initial . '\';">
            </a>';
    } else {
        // Display initials fallback
        $profileDisplayHtml = '
            <a href="profile.php" title="Welcome, ' . htmlspecialchars($currentUserName) . ' - Go to Profile">
                <div class="w-10 h-10 active-gradient rounded-full flex items-center justify-center font-bold text-white cursor-pointer super-shadow transition duration-300 hover:scale-105">
                    ' . $initial . '
                </div>
            </a>';
    }

    echo '<header class="bg-white p-4 sticky top-0 z-20 super-shadow rounded-2xl flex justify-between items-center h-20 mb-8 transform transition duration-300 hover:shadow-2xl">
        <div class="flex items-center space-x-2 text-xl font-extrabold text-gray-900 min-w-[200px]">
            <img src="assets/logo.png"
                alt="CLASS KEY Logo" 
                onerror="this.onerror=null;this.src=\'https://placehold.co/150x50/F4F7F9/1D206F?text=CLASS+LOGO\'"
                class="h-10 w-auto object-contain rounded-md">
        </div>
        <h1 class="flex-1 text-center text-3xl font-extrabold text-gray-900 truncate mx-4">CLASS KEY</h1>
        <div class="flex items-center space-x-4 min-w-[200px] justify-end">
            <button title="Notifications" class="w-10 h-10 rounded-full flex items-center justify-center text-gray-500 bg-gray-100 hover:bg-gray-200 hover:text-indigo-600 transition duration-200 shadow-inner">
                <i class="fa-solid fa-bell text-xl"></i>
            </button>
            ' . $profileDisplayHtml . '
        </div>
    </header>';
}

/**
 * Renders a set of admin navigation cards.
 * @param array $items Array of card data.
 */
function render_admin_cards($items) {
    echo '<div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">';
    foreach ($items as $item) {
        echo '<a href="' . htmlspecialchars($item['url']) . '" class="block">
            <div class="bg-white rounded-xl overflow-hidden shadow-lg hover:shadow-xl transition duration-300 transform hover:scale-[1.02]">
                <div class="p-6 flex flex-col items-center justify-center text-center">
                    <div class="text-indigo-600 mb-4">
                        <i class="' . htmlspecialchars($item['icon']) . ' text-5xl"></i>
                    </div>
                    <h3 class="text-xl font-bold text-gray-900 mb-1">' . htmlspecialchars($item['title']) . '</h3>
                    <p class="text-sm text-gray-500">' . htmlspecialchars($item['description']) . '</p>
                </div>
            </div>
        </a>';
    }
    echo '</div>';
}

/**
 * Renders the closing tags for the main content area.
 */
function render_main_content_end() {
    echo '</main></div>';
}
?>