<?php
// myclass.php - Handles display of enrolled lesson packs and lessons for a student.

require_once 'config.php';
require_login();

// Redirect admin users away from the student-specific page
if (is_admin()) {
    header('Location: dashboard.php');
    exit();
}

// --- Dynamic Variables from URL ---
$studentId = $_SESSION['user_id'];
$selectedPackId = $_GET['pack_id'] ?? null;
$selectedLessonId = $_GET['lesson_id'] ?? null; // NOTE: No longer used for on-page playback
$message = $_GET['msg'] ?? '';

// Fetch all granted lesson packs for the logged-in student.
$stmt = $pdo->prepare("
    SELECT 
        lp.id, 
        lp.title,
        lp.thumbnail_url,
        lp.description,
        (SELECT COUNT(id) FROM lessons WHERE pack_id = lp.id) as lesson_count
    FROM student_packs sp
    JOIN lesson_packs lp ON sp.pack_id = lp.id
    WHERE sp.student_id = ?
    ORDER BY lp.title
");
$stmt->execute([$studentId]);
$packs = $stmt->fetchAll();

$lessons = [];
// Default title is "My Classes" (for the card grid view)
$packTitle = 'My Classes'; 
// REMOVED: $youtubeEmbedUrl and related video logic are now in player.php

if ($selectedPackId) {
    // Check if student has access to the selected pack
    $stmt = $pdo->prepare("SELECT COUNT(id) FROM student_packs WHERE student_id = ? AND pack_id = ?");
    $stmt->execute([$studentId, $selectedPackId]);
    $hasAccess = $stmt->fetchColumn() > 0;

    if ($hasAccess) {
        // FETCH LESSONS: Added 'thumbnail_path' for custom thumbnails.
        $stmt = $pdo->prepare("SELECT id, title, description, youtube_link, thumbnail_path FROM lessons WHERE pack_id = ? ORDER BY id");
        $stmt->execute([$selectedPackId]);
        $lessons = $stmt->fetchAll();

        $packStmt = $pdo->prepare("SELECT title FROM lesson_packs WHERE id = ?");
        $packStmt->execute([$selectedPackId]);
        // Update title to the specific pack name for the lessons view
        $packTitle = $packStmt->fetchColumn() ?: 'Unknown Pack';

    } else {
        // If access denied, revert to showing the pack card view and display an error
        $message = 'Access denied to this lesson pack.';
        $selectedPackId = null;
        // Keep $packTitle as 'My Classes'
    }
}

// Ensure layout functions are available
require_once 'layouts.php'; 

// Render the page components
render_header('My Classes');
render_sidebar($_SESSION['user_role'], basename(__FILE__));
// Use the dynamic $packTitle for the topbar
render_topbar(htmlspecialchars($packTitle)); 
?>

<style>
/* Custom style for the pack/lesson image container - Increased height for visual dominance */
.pack-image-container {
    height: 250px; 
    background-size: cover;
    background-position: center;
}
/* Utility to restrict text lines, used in card descriptions */
.line-clamp-3 {
    display: -webkit-box;
    -webkit-box-orient: vertical;
    overflow: hidden;
    -webkit-line-clamp: 3;
}
/* REMOVED: .video-responsive styles are no longer needed on this page */
</style>

<div class="space-y-8">
    <?php if ($message): ?>
        <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-lg" role="alert"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if (!$selectedPackId): ?>
        <!-- SCENARIO A: Show all Lesson Pack Cards (UNCHANGED) -->
        <h2 class="text-2xl font-extrabold text-gray-900 mb-6">Your Enrolled Classes</h2>
        
        <?php if ($packs): ?>
            <!-- Responsive Card Grid -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                <?php foreach ($packs as $pack): ?>
                    <?php 
                        $has_thumbnail = !empty($pack['thumbnail_url']);
                        // Set style: use background image URL if available, otherwise use a placeholder
                        $header_style = $has_thumbnail 
                            ? "background-image: url('" . htmlspecialchars($pack['thumbnail_url']) . "');" 
                            : "background-color: #f3f4f6; background-image: url('https://placehold.co/600x250/f3f4f6/333333?text=No+Image+Available');"; 
                    ?>
                    <!-- Pack Card -->
                    <a href="?pack_id=<?php echo $pack['id']; ?>"
                       class="group block bg-white rounded-2xl shadow-xl hover:shadow-2xl transition-all duration-300 transform hover:scale-[1.02] overflow-hidden cursor-pointer border-4 border-transparent hover:border-indigo-500">
                        
                        <!-- Visual Header Area -->
                        <div class="pack-image-container relative flex flex-col justify-end" 
                             style="<?php echo $header_style; ?>">
                            <!-- Image area -->
                        </div>
                        
                        <!-- Details Area -->
                        <div class="p-4 pt-3">
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm font-semibold text-teal-600 px-2 py-0.5 rounded-full bg-teal-50">Class Pack</span>
                                <span class="text-green-600 font-bold flex items-center">
                                    <!-- Active Icon -->
                                    <svg class="w-5 h-5 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                    Active
                                </span>
                            </div>

                            <!-- Pack Title -->
                            <h3 class="text-xl font-extrabold text-gray-900 mb-3 leading-snug">
                                <?php echo htmlspecialchars($pack['title']); ?>
                            </h3>
                            
                            <?php if (!empty($pack['description'])): ?>
                            <!-- Dynamic Description: Uses nl2br to render line breaks -->
                            <div class="text-sm text-gray-700 mb-4 whitespace-pre-wrap line-clamp-3">
                                <?php echo nl2br(htmlspecialchars($pack['description'])); ?>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Card Footer (Progress and Video Count) -->
                            <div class="flex justify-between items-center text-sm font-medium pt-3 border-t mt-4">
                                <!-- Progress/Time Placeholder -->
                                <div class="flex items-center text-gray-500">
                                    <svg class="w-4 h-4 text-gray-400 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                                    None
                                </div>
                                <!-- Video Count -->
                                <div class="flex items-center text-gray-600">
                                    <svg class="w-4 h-4 text-indigo-500 mr-1" fill="currentColor" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path d="M2 6a2 2 0 012-2h12a2 2 0 012 2v8a2 2 0 01-2 2H4a2 2 0 01-2-2V6zm4.5 5a.5.5 0 00-.5.5v2a.5.5 0 00.771.419l1.5-1A.5.5 0 008 12.5v-2a.5.5 0 00-.771-.419l-1.5 1z"></path></svg>
                                    <span class="font-bold text-indigo-600"><?php echo $pack['lesson_count'] ?? 0; ?> Videos</span>
                                </div>
                            </div>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="bg-blue-100 border-l-4 border-blue-500 text-blue-700 p-4 rounded-lg" role="alert">
                You have not purchased any lesson packs yet.
                <a href="buyclass.php" class="text-indigo-600 hover:text-indigo-800 font-medium mt-2 block">Go to Buy Classes &rarr;</a>
            </div>
        <?php endif; ?>

    <?php else: ?>
        <!-- SCENARIO B: Show Lessons for the Selected Pack -->
        <div class="flex items-center justify-between pb-4 border-b mb-6">
            <h2 class="text-2xl font-extrabold text-gray-900"><?php echo htmlspecialchars($packTitle); ?> Lessons</h2>
            <!-- Back button to return to the card grid view -->
            <a href="myclass.php" class="text-indigo-600 hover:text-indigo-800 font-medium flex items-center p-2 rounded-lg transition-colors bg-indigo-50 hover:bg-indigo-100">
                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 mr-1" viewBox="0 0 20 20" fill="currentColor">
                    <path fill-rule="evenodd" d="M9.707 16.707a1 1 0 01-1.414 0l-6-6a1 1 0 010-1.414l6-6a1 1 0 011.414 1.414L5.414 9H17a1 1 0 110 2H5.414l4.293 4.293a1 1 0 010 1.414z" clip-rule="evenodd" />
                </svg>
                Back to All Classes
            </a>
        </div>
        
        <div class="bg-indigo-100 border-l-4 border-indigo-500 text-indigo-700 p-4 rounded-lg mb-6" role="alert">
            වීඩියෝව සම්පූර්ණ තිරයේ නැරඹීම සඳහා පහත පාඩම් මත ක්ලික් කරන්න. (Click on the lessons below to watch the video in a full screen.)
        </div>

        
        <?php if ($lessons): ?>
            <!-- Responsive Card Grid for Lessons -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                <?php foreach ($lessons as $lesson): ?>
                    <?php 
                        // CUSTOM THUMBNAIL LOGIC: Use thumbnail_path if available.
                        $has_thumbnail = !empty($lesson['thumbnail_path']);
                        $image_url = $has_thumbnail
                            ? htmlspecialchars($lesson['thumbnail_path'])
                            // Fallback placeholder image using the lesson title
                            : 'https://placehold.co/600x250/374151/ffffff?text=' . urlencode(substr(htmlspecialchars($lesson['title']), 0, 30));
                        
                        // UPDATED LINK: Now points to player.php and opens in a new tab for a dedicated viewing experience.
                        $lesson_link = "player.php?pack_id=" . htmlspecialchars($selectedPackId) . "&lesson_id=" . htmlspecialchars($lesson['id']);
                        
                        // Highlight logic is now removed as no video is playing on this page.
                        $action_text = 'Play Video in New Window';
                    ?>
                    <!-- Lesson Card - Links LOCALLY to load the video player -->
                    <a href="<?php echo $lesson_link; ?>" target="_blank"
                       class="group block bg-white rounded-2xl shadow-xl hover:shadow-2xl transition-all duration-300 transform hover:scale-[1.02] overflow-hidden cursor-pointer border-4 border-transparent hover:border-indigo-500">
                        
                        <!-- Visual Header Area -->
                        <div class="pack-image-container relative flex flex-col justify-end" 
                             style="background-image: url('<?php echo $image_url; ?>'); background-size: cover; background-position: center;">
                            <!-- Play button overlay indicator -->
                            <div class="absolute inset-0 flex items-center justify-center bg-black bg-opacity-30 group-hover:bg-opacity-50 transition-opacity">
                                <svg class="w-16 h-16 text-white opacity-80 group-hover:opacity-100 transition-opacity" fill="currentColor" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM9.555 7.168A1 1 0 008 8v4a1 1 0 001.555.832l3-2a1 1 0 000-1.664l-3-2z" clip-rule="evenodd"></path></svg>
                            </div>
                        </div>
                        
                        <!-- Details Area -->
                        <div class="p-4 pt-3">
                            <!-- Lesson Title -->
                            <h3 class="text-xl font-extrabold text-gray-900 mb-2 leading-snug line-clamp-2">
                                <?php echo htmlspecialchars($lesson['title']); ?>
                            </h3>
                            
                            <!-- Lesson Description -->
                            <p class="text-sm text-gray-700 mb-4 line-clamp-3">
                                <?php echo htmlspecialchars($lesson['description']); ?>
                            </p>
                            
                            <!-- Card Footer (Action Indicator) -->
                            <div class="flex justify-end items-center text-sm font-medium pt-3 border-t mt-4">
                                <div class="flex items-center text-indigo-600 font-bold">
                                    <svg class="w-4 h-4 mr-1" fill="currentColor" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path d="M10 12a2 2 0 100-4 2 2 0 000 4z"></path><path fill-rule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clip-rule="evenodd"></path></svg>
                                    <?php echo $action_text; ?>
                                </div>
                            </div>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="bg-yellow-100 border-l-4 border-yellow-500 text-yellow-700 p-4 rounded-lg" role="alert">
                No lessons have been added to this pack yet.
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<?php render_main_content_end(); ?>