<?php
// This file assumes config.php and helpers.php are included before it.
// Functions are suffixed with _2 to avoid conflict with layout.php

// Define a constant for App Name if not already defined (assuming it's usually defined in config.php)
if (!defined('APP_NAME')) {
    define('APP_NAME', 'CLASS KEY');
}

/**
 * Renders the HTML head, loads Tailwind, Inter font, and initial body styling for layout 2.
 * @param string $title The page title.
 */
function render_header_2($title) {
    $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . htmlspecialchars($title) . ' | ' . APP_NAME . '</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: "Inter", sans-serif; background-color: #f4f7f9; }
        
        /* Layout 2 Specific Styles */
        .wide-sidebar {
            width: 250px; /* Fixed width sidebar for desktop */
            transition: transform 0.3s ease-out;
            box-shadow: 2px 0 10px rgba(0, 0, 0, 0.1);
            background-color: #1e293b; /* Slate 800 */
        }
        .main-content {
            margin-left: 250px; /* Push content over on desktop */
            transition: margin-left 0.3s ease-out;
            min-height: 100vh;
        }
        
        /* Mobile: Hide sidebar by default, adjust content margin */
        @media (max-width: 1023px) {
            .wide-sidebar {
                transform: translateX(-100%);
                position: fixed;
                height: 100vh;
                z-index: 40;
            }
            .main-content {
                margin-left: 0;
            }
            .sidebar-open .wide-sidebar {
                transform: translateX(0);
            }
            .sidebar-open .backdrop {
                display: block;
            }
        }

        .backdrop {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            z-index: 30;
        }

        /* Active/Hover Styles */
        .sidebar-link {
            transition: all 0.2s;
            border-radius: 0.5rem;
        }
        .sidebar-link.active {
            background-color: #4f46e5; /* Indigo 600 */
            font-weight: 600;
            color: white;
            box-shadow: 0 4px 10px rgba(79, 70, 229, 0.3);
        }
        .sidebar-link:not(.active):hover {
            background-color: #334155; /* Slate 700 */
        }
    </style>
    <script>
        // Toggle function for the mobile sidebar drawer
        function toggleSidebar() {
            document.body.classList.toggle("sidebar-open");
        }
    </script>
</head>
<body class="overflow-x-hidden">
    <div class="backdrop lg:hidden" onclick="toggleSidebar()"></div>
    <div class="flex h-screen">';
    echo $html;
}

/**
 * Renders the Wide Fixed Sidebar and the Mobile Off-Canvas Drawer.
 * @param string $userRole 'admin' or any other role.
 * @param string $currentPage The current PHP file name (e.g., 'dashboard.php').
 */
function render_sidebar_2($userRole, $currentPage) {
    $menu = [];

    // Determine the menu items based on the user role
    if ($userRole === 'admin') {
        $menu = [
            'Dashboard' => ['url' => 'dashboard.php', 'icon' => 'fa-solid fa-chart-line'],
            'Lesson Packs' => ['url' => 'addpack.php', 'icon' => 'fa-solid fa-box-open'],
            'Add Lessons' => ['url' => 'addclass.php', 'icon' => 'fa-solid fa-graduation-cap'],
            'Add Notice' => ['url' => 'addnotice.php', 'icon' => 'fa-solid fa-bullhorn'],
            'User Management' => ['url' => 'usermanage.php', 'icon' => 'fa-solid fa-users'],
        ];
        $logoutItem = ['url' => 'index.php?logout=true', 'icon' => 'fa-solid fa-arrow-right-from-bracket', 'name' => 'Logout'];

    } else { // Student/Default Role
        $menu = [
            'Dashboard' => ['url' => 'studentdash.php', 'icon' => 'fa-solid fa-chart-line'],
            'My Classes' => ['url' => 'myclass.php', 'icon' => 'fa-solid fa-book'],
            'Buy Classes' => ['url' => 'buyclass.php', 'icon' => 'fa-solid fa-graduation-cap'],
            'Notices' => ['url' => 'notices.php', 'icon' => 'fa-solid fa-bell'],
            'Profile' => ['url' => 'profile.php', 'icon' => 'fa-solid fa-user-circle'],
        ];
        $logoutItem = ['url' => 'index.php?logout=true', 'icon' => 'fa-solid fa-power-off', 'name' => 'Logout'];
    }
    
    // --- Wide Fixed Sidebar (Desktop/Mobile Drawer) ---
    $sidebarHtml = '
        <aside class="wide-sidebar fixed lg:relative flex-shrink-0 flex flex-col justify-between h-full p-4 z-40">
            <!-- Logo/App Name -->
            <div class="mb-8 p-3">
                <h1 class="text-3xl font-extrabold text-indigo-400">CLASS<span class="text-white">KEY</span></h1>
                <p class="text-xs text-slate-400 mt-1">Version 2.0</p>
            </div>

            <!-- Navigation Links -->
            <nav class="flex-1 space-y-2 overflow-y-auto">
                <p class="text-xs font-semibold text-slate-500 uppercase tracking-wider px-3 pt-4 pb-2">Navigation</p>';

    foreach ($menu as $name => $item) {
        $isActive = (basename($item['url']) === basename($currentPage));
        $activeClass = $isActive ? ' active' : ' text-slate-300 hover:text-white';

        $sidebarHtml .= '
                <a href="' . htmlspecialchars($item['url']) . '"
                    class="sidebar-link flex items-center p-3 text-sm' . $activeClass . '">
                    <i class="' . htmlspecialchars($item['icon']) . ' w-6 h-6 mr-3 text-lg"></i>
                    ' . htmlspecialchars($name) . '
                </a>';
    }

    // Sidebar Logout Button
    $sidebarHtml .= '
            </nav>
            <div class="mt-8">
                <a href="' . htmlspecialchars($logoutItem['url']) . '" title="' . htmlspecialchars($logoutItem['name']) . '"
                    class="flex items-center justify-center p-3 w-full bg-red-600 text-white rounded-lg hover:bg-red-700 transition duration-300 shadow-xl">
                    <i class="fa-solid fa-power-off mr-2"></i>
                    ' . htmlspecialchars($logoutItem['name']) . '
                </a>
            </div>
        </aside>';
    
    echo $sidebarHtml;
}

/**
 * Renders the top bar for layout 2 (sits above the main content area only).
 * @param string $title The main page title (e.g., Dashboard).
 * @param string|null $imageUrl URL for the user profile picture.
 */
function render_topbar_2($title, $imageUrl = null) {
    $currentUserName = $_SESSION['user_name'] ?? 'Guest';
    $initial = strtoupper(substr($currentUserName, 0, 1));

    if ($imageUrl) {
        $profileDisplayHtml = '
            <a href="profile.php" title="Welcome, ' . htmlspecialchars($currentUserName) . ' - Go to Profile">
                <img src="' . htmlspecialchars($imageUrl) . '" alt="Profile Picture"
                    class="w-10 h-10 rounded-full object-cover border-2 border-indigo-600 cursor-pointer transition duration-300 hover:scale-105"
                    onerror="this.onerror=null;this.src=\'https://placehold.co/40x40/2563EB/ffffff?text=' . $initial . '\';">
            </a>';
    } else {
        $profileDisplayHtml = '
            <a href="profile.php" title="Welcome, ' . htmlspecialchars($currentUserName) . ' - Go to Profile">
                <div class="w-10 h-10 bg-indigo-600 rounded-full flex items-center justify-center font-bold text-white cursor-pointer transition duration-300 hover:scale-105">
                    ' . $initial . '
                </div>
            </a>';
    }
    
    echo '
    <div class="main-content flex-1 flex flex-col">
        <header class="bg-white p-4 sticky top-0 z-20 shadow-md flex justify-between items-center h-20">
            <!-- Mobile Menu Toggle -->
            <button onclick="toggleSidebar()" class="p-2 mr-4 text-gray-600 rounded-lg lg:hidden hover:bg-gray-100">
                <i class="fa-solid fa-bars text-xl"></i>
            </button>
            
            <h2 class="text-2xl font-extrabold text-gray-900 truncate flex-1">' . htmlspecialchars($title) . '</h2>
            
            <div class="flex items-center space-x-4">
                <button title="Notifications" class="w-10 h-10 rounded-full flex items-center justify-center text-gray-500 bg-gray-100 hover:bg-gray-200 hover:text-indigo-600 transition duration-200">
                    <i class="fa-solid fa-bell text-xl"></i>
                </button>
                ' . $profileDisplayHtml . '
            </div>
        </header>
        
        <!-- Main Content Area Start -->
        <main class="flex-1 p-6 lg:p-10">';
}

/**
 * Renders attractive admin action cards (reused for Layout 2).
 * @param array $items Array of cards, each with 'title', 'description', 'icon', and 'url'.
 */
function render_admin_cards_2($items) {
    echo '<div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">';
    foreach ($items as $item) {
        echo '<a href="' . htmlspecialchars($item['url']) . '" class="block">
            <div class="bg-white rounded-xl overflow-hidden shadow-lg hover:shadow-xl transition duration-300 transform hover:scale-[1.02] border-b-4 border-indigo-500">
                <div class="p-6 flex flex-col items-center justify-center text-center">
                    <div class="text-indigo-600 mb-4 bg-indigo-100 p-3 rounded-xl">
                        <i class="' . htmlspecialchars($item['icon']) . ' text-4xl"></i>
                    </div>
                    <h3 class="text-xl font-bold text-gray-900 mb-1">' . htmlspecialchars($item['title']) . '</h3>
                    <p class="text-sm text-gray-500">' . htmlspecialchars($item['description']) . '</p>
                </div>
            </div>
        </a>';
    }
    echo '</div>';
}

/**
 * Closes the main content tags for layout 2.
 */
function render_main_content_end_2() {
    // Closes the main tag and the main-content wrapper div
    echo '</main></div></div>'; // Close main, main-content, and the initial flex container
}

/**
 * Renders the closing HTML tags for layout 2.
 */
function render_footer_2() {
    echo '</body></html>';
}
?>